// $Id: ContourMapM.nc,v 1.3.4.3 2005/03/15 20:08:12 wwxue Exp $

/*
* Copyright (c) 2006, The Hong Kong University of Science and Technology (HKUST)
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without 
* modification, are permitted provided that the following conditions 
* are met:
*
*   * Redistributions of source code must retain the above copyright notice, 
*     this list of conditions,the authors and the following disclaimer.
*   * Redistributions in binary form must reproduce the above copyright notice,
*     this list of conditions,the authors and the following disclaimer in
*     the documentation and/or other materials provided with the distribution.
*   * Neither the name of the university nor the names of its 
*     contributors may be used to endorse or promote products derived from 
*     this software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
* ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
* POSSIBILITY OF SUCH DAMAGE.
*
* Author: Wenwei Xue (wwxue@cs.ust.hk)
* Date last modified:  03/15/06
*/

includes Aggregates;
//includes TinyDB;

module ContourMapM {
	provides {
		interface Aggregate;
	}
	uses {
		interface Leds;
	}
}

implementation {

	typedef struct {	
		uint8_t x;		
		uint8_t y;
	} Vertex;  // 2 bytes
	
	/** Now we simplify the problem and assume all contour regions are rectangles without holes */
	#define MAX_VERTEX_NUM 2
	
	typedef struct {
		Vertex vertices[MAX_VERTEX_NUM];
	} ContourRegionBoundary;  // 4 bytes
	
	#define MAX_BOUNDARY_NUM 1
	
	typedef struct {
		//int16_t w[3];
		int8_t A[10];
		int8_t b[6];
	} RegressionModel;  // 16 bytes
	
	typedef struct {
		ContourRegionBoundary boundaries[MAX_BOUNDARY_NUM];
		RegressionModel regression;
		uint8_t epsilon;  // must divide by 100 when it is used
	} ContourRegion;  // 21 bytes
	
	/** In the worse case, this is equal to the number of nodes in the network */
	#define MAX_REGION_NUM 3
	
	typedef struct {
		uint8_t regionNum;
		ContourRegion regions[MAX_REGION_NUM];		
	} ContourMap;  // 64 bytes
	
	#define CELL_LEN 20
	
	uint8_t loc_x = 0, loc_y = 0;
	uint8_t x[3], y[3];
	uint8_t errorBound = 20, mergingLimit = 100;
	ContourMap *mymap;
	
	void initializeLocation () {
		if (TOS_LOCAL_ADDRESS == 1)
		  {loc_x = 10; loc_y = 10;}
		else
		  if (TOS_LOCAL_ADDRESS == 2)
		    {loc_x = 30; loc_y = 10;}
		  else
		    if (TOS_LOCAL_ADDRESS == 3)
		      {loc_x = 10; loc_y = 30;}
		    else
		      if (TOS_LOCAL_ADDRESS == 4)
		        {loc_x = 30; loc_y = 30;}
		      else
		        if (TOS_LOCAL_ADDRESS == 5)
		          {loc_x = 50; loc_y = 10;}
		        else
		          if (TOS_LOCAL_ADDRESS == 6)
		            {loc_x = 50; loc_y = 30;}
		          else
		            if (TOS_LOCAL_ADDRESS == 7)
		              {loc_x = 10; loc_y = 50;}
		            else 
		              if (TOS_LOCAL_ADDRESS == 8)
		                {loc_x = 30; loc_y = 50;}
		              else
		                if (TOS_LOCAL_ADDRESS == 9)
		                  {loc_x = 50; loc_y = 50;}
		                else 
		                  if (TOS_LOCAL_ADDRESS == 10)
		                    {loc_x = 70; loc_y = 10;}
		                  else 
		                    if (TOS_LOCAL_ADDRESS == 11)
		                      {loc_x = 70; loc_y = 30;}
		                    else
		                      if (TOS_LOCAL_ADDRESS == 12)
		                        {loc_x = 70; loc_y = 50;}	      
	}	
	
	void constructRegressionModel (RegressionModel *regression, int16_t *w) {	      
		int16_t d, d1, d2;
		int16_t *A, *b;
		
		A = (int16_t*)regression->A;
		b = (int16_t*)regression->b;
		
		d = 1*A[2] - A[0]*A[0];
		d1 = b[2]*d + b[0]*A[0]*A[3] - b[1]*1*A[3] - b[0]*A[2]*A[1] + b[1]*A[0]*A[1];
		d2 = A[0]*A[3]*A[1] - A[1]*A[2]*A[1] - 1*A[3]*A[3] + A[1]*A[0]*A[3] + A[4]*d;
		
		w[2] = d1;
		if (d2 != 0)
		  w[2] /= d2;
		else
		  w[2] = 0;
		  
		w[0] = w[2]*(A[0]*A[3] - A[1]*A[2]) + (b[0]*A[2] - b[1]*A[0]);
		if (d != 0)
		  w[0] /= d;
		else
		  w[0] = 0;
		  
		w[1] = w[2]*(1*A[3] - A[1]*A[0]) + (b[0]*A[0] - b[1]*1);
		if (d != 0)
		  w[1] /= (-d);
		else
		  w[1] = 0;
		  
		if (w[1] == -0)
		  w[1] = 0;		
	}
	
	bool checkOverlapping(ContourRegion *region1, ContourRegion *region2) {
		if (region2->boundaries[0].vertices[0].y == region1->boundaries[0].vertices[0].y)
		  if (region2->boundaries[0].vertices[1].y == region1->boundaries[0].vertices[1].y)
		    {
		    	if (region2->boundaries[0].vertices[0].x - region1->boundaries[0].vertices[0].x <= CELL_LEN)
		    	  if (region2->boundaries[0].vertices[1].x - region1->boundaries[0].vertices[1].x <= CELL_LEN)
		    	    return TRUE;
		    	    
		    	if (region1->boundaries[0].vertices[0].x - region2->boundaries[0].vertices[0].x <= CELL_LEN)
		    	  if (region1->boundaries[0].vertices[1].x - region2->boundaries[0].vertices[1].x <= CELL_LEN)
		    	    return TRUE;
		    }	
		    
		if (region2->boundaries[0].vertices[0].x == region1->boundaries[0].vertices[0].x)
		  if (region2->boundaries[0].vertices[1].x == region1->boundaries[0].vertices[1].x)
		    {
		    	if (region2->boundaries[0].vertices[0].y - region1->boundaries[0].vertices[0].y <= CELL_LEN)
		    	  if (region2->boundaries[0].vertices[1].y - region1->boundaries[0].vertices[1].y <= CELL_LEN)
		    	    return TRUE;
		    	    
		    	if (region1->boundaries[0].vertices[0].y - region2->boundaries[0].vertices[0].y <= CELL_LEN)
		    	  if (region1->boundaries[0].vertices[1].y - region2->boundaries[0].vertices[1].y <= CELL_LEN)
		    	    return TRUE;
		    }
		    
		return FALSE;	
	}
	
	uint16_t computeVol (ContourRegion *region, int16_t *w) {
		int32_t d = (w[0] + w[1]*(region->boundaries[0].vertices[0].x + region->boundaries[0].vertices[1].x) + w[2]*(region->boundaries[0].vertices[0].y + region->boundaries[0].vertices[1].y));        	
	        if (d < 0) d = -d;
	        return CELL_LEN*CELL_LEN*d;
	}
	
        void computeRegionEpsilon(ContourRegion *region1, ContourRegion *region2, ContourRegion *region) {
        	int i;
        	uint16_t vol1, vol2; 
        	int16_t *A1, *A2, *A3, *b1, *b2, *b3;         	
        	int16_t w1[3], w2[3], w[3];
        	uint16_t epsilon;
        	
        	A1 = (int16_t*)region1->regression.A;
        	A2 = (int16_t*)region2->regression.A;
        	A3 = (int16_t*)region->regression.A;
        	for (i=0;i<5;i++)
        	  A3[i] = A1[i] + A2[i];     	
        	
        	b1 = (int16_t*)region1->regression.b;
        	b2 = (int16_t*)region2->regression.b;
        	b3 = (int16_t*)region->regression.b;
        	for (i=0;i<3;i++)
        	  b3[i] = b1[i] + b2[i];  
        	 
        	constructRegressionModel(&(region1->regression),w1);
        	constructRegressionModel(&(region2->regression),w2); 
        	constructRegressionModel(&(region->regression), w);
        	epsilon = 0;
        	 	  
        	vol1 = computeVol(region1, w);
        	vol2 = computeVol(region1, w1);
        	if (vol1 - vol2 > 0)
        	  epsilon += vol1 - vol2;
        	else
        	  epsilon += vol2 - vol1;        	
        	vol1 = computeVol(region2, w);
        	vol2 = computeVol(region2, w2);
        	if (vol1 - vol2 > 0)
        	  epsilon += vol1 - vol2;
        	else
        	  epsilon += vol2 - vol1;
        	vol1 = computeVol(region1, w1);
        	vol2 = computeVol(region2, w2);
        	epsilon += region1->epsilon*vol1 + region2->epsilon*vol2;
        	epsilon = epsilon*100;
        	epsilon /= vol1 + vol2;
        	if (epsilon%(vol1+vol2) != 0)
        	  epsilon += 1;
        	region->epsilon = epsilon;        	
        }
        
        /** Check if region2 can be merged into region1 */
        bool contourRegionMerging(ContourRegion *region1, ContourRegion *region2) {
        	
        	ContourRegion region;
        	
        	if (checkOverlapping(region1, region2) == FALSE)
        	  return FALSE;
        	  
        	computeRegionEpsilon(region1,region2, &region);
        	if (region.epsilon > errorBound)
        	  return FALSE;
        	region.boundaries[0].vertices[0] = region1->boundaries[0].vertices[0];
        	region.boundaries[0].vertices[1] = region1->boundaries[0].vertices[1];
        	if (region.boundaries[0].vertices[0].x > region2->boundaries[0].vertices[0].x)
        	  region.boundaries[0].vertices[0].x = region2->boundaries[0].vertices[0].x;
        	if (region.boundaries[0].vertices[0].y > region2->boundaries[0].vertices[0].y)
        	  region.boundaries[0].vertices[0].y = region2->boundaries[0].vertices[0].y;
        	if (region.boundaries[0].vertices[1].x < region2->boundaries[0].vertices[1].x)
        	  region.boundaries[0].vertices[1].x = region2->boundaries[0].vertices[1].x;
        	if (region.boundaries[0].vertices[1].y < region2->boundaries[0].vertices[1].y)
        	  region.boundaries[0].vertices[1].y = region2->boundaries[0].vertices[1].y;
        	
        	// merge the second region into the first one
        	*region1 = region;
        	  
        	return TRUE;
        }
	
	//we'll probably get rid of this later
	command result_t Aggregate.update(char *destdata, char* value, ParamList *params, ParamVals *paramValues) {
		
		int16_t v = *(int16_t*)value;		
		int i;
		int16_t *b;
		ContourMap *currentmap = (ContourMap *)destdata;		
		
		if (isRoot())
		  return SUCCESS;
				
		b = (int16_t*)mymap->regions[0].regression.b;
		b[0]=3*v;
		b[1]=v*(x[0]+x[1]+x[2]);
		b[2]=v*(y[0]+y[1]+y[2]);				
		
		for (i=0;i<currentmap->regionNum;i++)
		  if (contourRegionMerging(&currentmap->regions[i],&mymap->regions[0]))
		    break;		  			  
		if (i==currentmap->regionNum)
		  currentmap->regions[currentmap->regionNum++]=mymap->regions[0];
		
		// call Leds.redToggle();				
		
		return SUCCESS;
	}
	
	command result_t Aggregate.merge(char *destdata, char *mergedata, ParamList *params, ParamVals *paramValues) {
		
		ContourMap *currentmap = (ContourMap *)destdata;
		ContourMap *newmap = (ContourMap *)mergedata;
		int i,j;
		bool merged [newmap->regionNum];		
		
		for (i=0;i<MAX_REGION_NUM;i++)
		  merged[i] = FALSE;		
		
		for (i=0;i<currentmap->regionNum;i++)
		  for (j=0;j<newmap->regionNum;j++)
		    if (contourRegionMerging(&currentmap->regions[i],&newmap->regions[j]))
		    	  {merged[j] = TRUE; break;}
		    
		for (i=0;i<newmap->regionNum;i++)
		  if (merged[i] == FALSE)
		    currentmap->regions[currentmap->regionNum++] = newmap->regions[i];
		    
		//call Leds.redToggle();
		
		return SUCCESS;
	}

	//doubles as startEpoch right now? might separate the two
	command result_t Aggregate.init(char *data, ParamList *params, ParamVals *paramValues, bool isFirstTime){
		
		int16_t *A;
		
		//if (isFirstTime)
		  //{
		  	mymap = (ContourMap *)data;
		  	
		  	if (isRoot())
		  	  mymap->regionNum = 0;
		  	else
		  	  {
		  	  	initializeLocation();	
		
				mymap->regions[0].boundaries[0].vertices[0].x = loc_x - CELL_LEN / 2;
				mymap->regions[0].boundaries[0].vertices[0].y = loc_y - CELL_LEN / 2;
				mymap->regions[0].boundaries[0].vertices[1].x = loc_x + CELL_LEN / 2;
				mymap->regions[0].boundaries[0].vertices[1].y = loc_y + CELL_LEN / 2;
				mymap->regionNum = 1;
		
				x[0]=loc_x; y[0]=loc_y;
				x[1]=loc_x; y[1]=loc_y+5;
				x[2]=loc_x+5; y[2]=loc_y;			
				A = (int16_t*)mymap->regions[0].regression.A;
				A[0]=x[0]+x[1]+x[2];
				A[1]=y[0]+y[1]+y[2];
				A[2]=x[0]*x[0]+x[1]*x[1]+x[2]*x[2];
				A[3]=x[0]*y[0]+x[1]*y[1]+x[2]*y[2];
				A[4]=y[0]*y[0]+y[1]*y[1]+y[2]*y[2];							
			  }
		  //}
		  
		if (isFirstTime)
		  {
		    errorBound = getErrorBound(paramValues);
		    mergingLimit = getMergingLimit(paramValues);		  
		  }				
		
		return SUCCESS;
	}

	command uint16_t Aggregate.stateSize(ParamList *params, ParamVals *paramValues) {
		
		uint16_t size = sizeof(ContourMap);
		  
		return size;
	}

	command bool Aggregate.hasData(char *data, ParamList *params, ParamVals *paramValues) {		
		return TRUE;		
	}

	command TinyDBError Aggregate.finalize(char *data, char *result_buf, ParamList *params, ParamVals *paramValues) {
		
		return err_NoError;
	}	
	
	/*command AggregateProperties Aggregate.getProperties() {
		return 0;
	}*/
	
	
}
